/*
 * Decompiled with CFR 0.152.
 */
package com.amazon.opendistroforelasticsearch.knn.index;

import com.amazon.opendistroforelasticsearch.knn.index.KNNSettings;
import com.amazon.opendistroforelasticsearch.knn.index.v2011.KNNIndex;
import com.amazon.opendistroforelasticsearch.knn.plugin.stats.StatNames;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheStats;
import com.google.common.cache.RemovalCause;
import com.google.common.cache.RemovalNotification;
import java.io.Closeable;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.FileWatcher;
import org.elasticsearch.watcher.ResourceWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;
import org.elasticsearch.watcher.WatcherHandle;

public class KNNIndexCache
implements Closeable {
    public static String GRAPH_COUNT = "graph_count";
    private static Logger logger = LogManager.getLogger(KNNIndexCache.class);
    private static KNNIndexCache INSTANCE;
    private Cache<String, KNNIndexCacheEntry> cache;
    private ExecutorService executor = Executors.newSingleThreadExecutor();
    private AtomicBoolean cacheCapacityReached;
    private ResourceWatcherService resourceWatcherService;
    private static FileChangesListener KNN_INDEX_FILE_DELETED_LISTENER;

    private KNNIndexCache() {
        this.initCache();
    }

    public static void setResourceWatcherService(ResourceWatcherService resourceWatcherService) {
        KNNIndexCache.getInstance().resourceWatcherService = resourceWatcherService;
    }

    @Override
    public void close() {
        this.executor.shutdown();
    }

    public static synchronized KNNIndexCache getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new KNNIndexCache();
        }
        return INSTANCE;
    }

    private void initCache() {
        CacheBuilder cacheBuilder = CacheBuilder.newBuilder().recordStats().concurrencyLevel(1).removalListener(k -> this.onRemoval((RemovalNotification<String, KNNIndexCacheEntry>)k));
        if (((Boolean)KNNSettings.state().getSettingValue("knn.memory.circuit_breaker.enabled")).booleanValue()) {
            cacheBuilder.maximumWeight(KNNSettings.getCircuitBreakerLimit().getKb()).weigher((k, v) -> (int)((KNNIndexCacheEntry)v).getKnnIndex().getIndexSize());
        }
        if (((Boolean)KNNSettings.state().getSettingValue("knn.cache.item.expiry.enabled")).booleanValue()) {
            long expiryTime = ((TimeValue)KNNSettings.state().getSettingValue("knn.cache.item.expiry.minutes")).getMinutes();
            cacheBuilder.expireAfterAccess(expiryTime, TimeUnit.MINUTES);
        }
        this.cacheCapacityReached = new AtomicBoolean(false);
        this.cache = cacheBuilder.build();
    }

    public synchronized void rebuild() {
        logger.info("KNN Cache rebuilding.");
        this.executor.execute(() -> {
            this.cache.invalidateAll();
            this.initCache();
        });
    }

    private void onRemoval(RemovalNotification<String, KNNIndexCacheEntry> removalNotification) {
        KNNIndexCacheEntry knnIndexCacheEntry = (KNNIndexCacheEntry)removalNotification.getValue();
        knnIndexCacheEntry.getFileWatcherHandle().stop();
        this.executor.execute(() -> knnIndexCacheEntry.getKnnIndex().close());
        String esIndexName = ((KNNIndexCacheEntry)removalNotification.getValue()).getEsIndexName();
        String indexPathUrl = ((KNNIndexCacheEntry)removalNotification.getValue()).getIndexPathUrl();
        if (RemovalCause.SIZE == removalNotification.getCause()) {
            KNNSettings.state().updateCircuitBreakerSettings(true);
            this.setCacheCapacityReached(true);
        }
        logger.info("[KNN] Cache evicted. Key {}, Reason: {}", removalNotification.getKey(), (Object)removalNotification.getCause());
    }

    public KNNIndex getIndex(String key, String indexName) {
        try {
            KNNIndexCacheEntry knnIndexCacheEntry = (KNNIndexCacheEntry)this.cache.get((Object)key, () -> this.loadIndex(key, indexName));
            return knnIndexCacheEntry.getKnnIndex();
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    public List<KNNIndex> getIndices(List<String> segmentPaths, String indexName) {
        return segmentPaths.stream().map(segmentPath -> this.getIndex((String)segmentPath, indexName)).collect(Collectors.toList());
    }

    public CacheStats getStats() {
        return this.cache.stats();
    }

    public Map<String, Map<String, Object>> getIndicesCacheStats() {
        HashMap<String, Map<String, Object>> statValues = new HashMap<String, Map<String, Object>>();
        for (Map.Entry index : this.cache.asMap().entrySet()) {
            String indexName = ((KNNIndexCacheEntry)index.getValue()).getEsIndexName();
            statValues.putIfAbsent(indexName, new HashMap());
            ((Map)statValues.get(indexName)).put(GRAPH_COUNT, ((Map)statValues.get(indexName)).getOrDefault(GRAPH_COUNT, 0) + 1);
            ((Map)statValues.get(indexName)).putIfAbsent(StatNames.GRAPH_MEMORY_USAGE.getName(), this.getWeightInKilobytes(indexName));
            ((Map)statValues.get(indexName)).putIfAbsent(StatNames.GRAPH_MEMORY_USAGE_PERCENTAGE.getName(), this.getWeightAsPercentage(indexName));
        }
        return statValues;
    }

    protected Set<String> getGraphNamesForIndex(String indexName) {
        return this.cache.asMap().values().stream().filter(knnIndexCacheEntry -> indexName.equals(((KNNIndexCacheEntry)knnIndexCacheEntry).getEsIndexName())).map(rec$ -> ((KNNIndexCacheEntry)rec$).getIndexPathUrl()).collect(Collectors.toSet());
    }

    public Long getWeightInKilobytes() {
        return this.cache.asMap().values().stream().map(rec$ -> ((KNNIndexCacheEntry)rec$).getKnnIndex()).mapToLong(KNNIndex::getIndexSize).sum();
    }

    public Long getWeightInKilobytes(String indexName) {
        return this.cache.asMap().values().stream().filter(knnIndexCacheEntry -> indexName.equals(((KNNIndexCacheEntry)knnIndexCacheEntry).getEsIndexName())).map(rec$ -> ((KNNIndexCacheEntry)rec$).getKnnIndex()).mapToLong(KNNIndex::getIndexSize).sum();
    }

    public Float getWeightAsPercentage() {
        return Float.valueOf((float)(100L * this.getWeightInKilobytes()) / (float)KNNSettings.getCircuitBreakerLimit().getKb());
    }

    public Float getWeightAsPercentage(String indexName) {
        return Float.valueOf((float)(100L * this.getWeightInKilobytes(indexName)) / (float)KNNSettings.getCircuitBreakerLimit().getKb());
    }

    public Boolean isCacheCapacityReached() {
        return this.cacheCapacityReached.get();
    }

    public void setCacheCapacityReached(Boolean value) {
        this.cacheCapacityReached.set(value);
    }

    public void evictGraphFromCache(String indexFilePath) {
        logger.info("[KNN] " + indexFilePath + " invalidated explicitly");
        this.cache.invalidate((Object)indexFilePath);
    }

    public void evictAllGraphsFromCache() {
        logger.info("[KNN] All entries in cache invalidated explicitly");
        this.cache.invalidateAll();
    }

    public KNNIndexCacheEntry loadIndex(String indexPathUrl, String indexName) throws Exception {
        if (Strings.isNullOrEmpty((String)indexPathUrl)) {
            throw new IllegalStateException("indexPath is null while performing load index");
        }
        logger.debug("[KNN] Loading index: {}", (Object)indexPathUrl);
        Path indexPath = Paths.get(indexPathUrl, new String[0]);
        FileWatcher fileWatcher = new FileWatcher(indexPath);
        fileWatcher.addListener((Object)KNN_INDEX_FILE_DELETED_LISTENER);
        fileWatcher.init();
        KNNIndex knnIndex = KNNIndex.loadIndex(indexPathUrl, this.getQueryParams(indexName), KNNSettings.getSpaceType(indexName));
        WatcherHandle watcherHandle = this.resourceWatcherService.add((ResourceWatcher)fileWatcher);
        return new KNNIndexCacheEntry(knnIndex, indexPathUrl, indexName, watcherHandle);
    }

    private String[] getQueryParams(String indexName) {
        return new String[]{"efSearch=" + KNNSettings.getEfSearchParam(indexName)};
    }

    static {
        KNN_INDEX_FILE_DELETED_LISTENER = new FileChangesListener(){

            public void onFileDeleted(Path indexFilePath) {
                logger.debug("[KNN] Invalidated because file {} is deleted", (Object)indexFilePath.toString());
                KNNIndexCache.getInstance().cache.invalidate((Object)indexFilePath.toString());
            }
        };
    }

    private static class KNNIndexCacheEntry {
        private final KNNIndex knnIndex;
        private final String indexPathUrl;
        private final String esIndexName;
        private final WatcherHandle<FileWatcher> fileWatcherHandle;

        private KNNIndexCacheEntry(KNNIndex knnIndex, String indexPathUrl, String esIndexName, WatcherHandle<FileWatcher> fileWatcherHandle) {
            this.knnIndex = knnIndex;
            this.indexPathUrl = indexPathUrl;
            this.esIndexName = esIndexName;
            this.fileWatcherHandle = fileWatcherHandle;
        }

        private KNNIndex getKnnIndex() {
            return this.knnIndex;
        }

        private String getIndexPathUrl() {
            return this.indexPathUrl;
        }

        private String getEsIndexName() {
            return this.esIndexName;
        }

        private WatcherHandle<FileWatcher> getFileWatcherHandle() {
            return this.fileWatcherHandle;
        }
    }
}

